#include "IFileStream.h"
#include "IErrors.h"
#include "MTExceptions.h"

//	IFileStream::IFileStream()
//	
//	constructor
IFileStream::IFileStream()
:fref(0)
{
	
}

//	IFileStream::~IFileStream()
//	
//	destructor
IFileStream::~IFileStream()
{
	CloseFile();
}

//	void IFileStream::OpenFile(FSSpec * theFile)
//	
//	opens a file for streaming
//	
//	theFile = pointer to file spec
void IFileStream::OpenFile(FSSpec * theFile)
{
	OSErr	theErr;
	
	theErr = FSpOpenDF(theFile, fsRdWrPerm, &fref);
	if(theErr)
		throw MTOSFileException(theErr, "Error opening file");
	
	GetFileInfo();
}

//	void IFileStream::CreateFile(FSSpec * theFile, OSType creator, OSType type)
//	
//	creates a file and opens it for streaming
//	
//	theFile = pointer to file spec
//	creator = file creator
//	type = file type
void IFileStream::CreateFile(FSSpec * theFile, OSType creator, OSType type)
{
	OSErr	theErr;
	
	theErr = FSpCreate(theFile, creator, type, smSystemScript);
	if(theErr)
		throw MTOSFileException(theErr, "Error creating file");
	
	theErr = FSpOpenDF(theFile, fsRdWrPerm, &fref);
	if(theErr)
		throw MTOSFileException(theErr, "Error opening file");
	
	length = 0;
	bytesLeft = 0;
}

//	void IFileStream::CloseFile(void)
//	
//	closes the file
void IFileStream::CloseFile(void)
{
	if(fref)
	{
		OSErr	theErr;
		
		theErr = FSClose(fref);
		if(theErr)
			throw MTOSFileException(theErr, "Error closing file");
		
		fref = 0;
	}
}

//	UInt32 IFileStream::GetPosition(void)
//	
//	returns offset into the file
//	
//	returns offset
UInt32 IFileStream::GetPosition(void)
{
	OSErr	theErr;
	long	position;
	
	theErr = GetFPos(fref, &position);
	if(theErr)
		throw MTOSFileException(theErr, "Error getting file position");
	
	return position;
}

//	void IFileStream::SetPosition(UInt32 position)
//	
//	sets offset into the file
//	
//	position = offset
void IFileStream::SetPosition(UInt32 position)
{
	DEBUG_ASSERT(position < length);
	
	OSErr	theErr;
	
	theErr = SetFPos(fref, fsFromStart, position);
	if(theErr)
	{
		if(theErr == eofErr)
		{
			throw MTRangeException("Tried to seek past the end of the file");
		}
		else
		{
			throw MTOSFileException(theErr, "Error setting file position");
		}
	}
	
	bytesLeft = length - position;
}

//	void IFileStream::Skip(UInt32 bytesToSkip)
//	
//	skips bytes
//	
//	bytesToSkip = number of bytes to skip
void IFileStream::Skip(UInt32 bytesToSkip)
{
	if(bytesLeft < bytesToSkip)
		throw MTRangeException("Reached end of file");
	
	OSErr	theErr;
	
	theErr = SetFPos(fref, fsFromMark, bytesToSkip);
	if(theErr)
		throw MTOSFileException(theErr, "Error setting file position");
	
	bytesLeft -= bytesToSkip;
}

//	void IFileStream::ReadBuffer(UInt8 * buf, UInt32 size)
//	
//	reads from a file into a buffer
//	
//	buf = buffer to read to
//	size = amount to read
void IFileStream::ReadBuffer(UInt8 * buf, UInt32 size)
{
	if(bytesLeft < size)
		throw MTRangeException("Reached end of file");
	
	OSErr	theErr;
	long	bytes = size;
	
	theErr = FSRead(fref, &bytes, buf);
	if(theErr)
		throw MTOSFileException(theErr, "Error reading from file");
	
	bytesLeft -= size;
}

//	void IFileStream::WriteBuffer(UInt8 * buf, UInt32 size)
//	
//	writes from a buffer into a file
//	
//	buf = buffer to write to
//	size = amount to write
void IFileStream::WriteBuffer(UInt8 * buf, UInt32 size)
{
	OSErr	theErr;
	long	bytes = size;
	
	theErr = FSWrite(fref, &bytes, buf);
	if(theErr)
		throw MTOSFileException(theErr, "Error writing to file");
	
	bytesLeft += size;
	length += size;
}

//	void IFileStream::GetFileInfo(void)
//	
//	sets up internal size fields
void IFileStream::GetFileInfo(void)
{
	OSErr	theErr;
	long	size, position;
	
	theErr = GetEOF(fref, &size);
	if(theErr)
		throw MTOSFileException(theErr, "Error getting file size");
	
	theErr = GetFPos(fref, &position);
	if(theErr)
		throw MTOSFileException(theErr, "Error getting file position");
	
	length = size;
	bytesLeft = size - position;
}